-- PyrosparkCore - Client Functions
-- Exported functions for use by other resources

-- ============================================================================
--                          EXPORTED FUNCTIONS
-- ============================================================================

-- Get player coordinates
-- Usage: local coords = exports['PS-PyrosparkCore']:GetPlayerCoords()
-- Returns: vector3(x, y, z)
exports('GetPlayerCoords', function()
    local playerPed = PlayerPedId()
    return GetEntityCoords(playerPed)
end)

-- Get player coordinates with heading
-- Usage: local coords = exports['PS-PyrosparkCore']:GetPlayerCoordsHeading()
-- Returns: vector4(x, y, z, heading)
exports('GetPlayerCoordsHeading', function()
    local playerPed = PlayerPedId()
    local coords = GetEntityCoords(playerPed)
    local heading = GetEntityHeading(playerPed)
    return vector4(coords.x, coords.y, coords.z, heading)
end)

-- Helper function to find closest vehicle within radius (checks all vehicle types)
local function GetClosestVehicleInRadius(coords, radius)
    local closestVehicle = nil
    local closestDistance = radius + 1.0

    -- Get all vehicles and check which ones are within radius
    local vehicles = GetGamePool('CVehicle')

    for _, vehicle in ipairs(vehicles) do
        local vehCoords = GetEntityCoords(vehicle)
        local distance = #(coords - vehCoords)

        if distance <= radius and distance < closestDistance then
            closestDistance = distance
            closestVehicle = vehicle
        end
    end

    return closestVehicle
end

-- Delete nearest vehicle
-- Usage: local success = exports['PS-PyrosparkCore']:DeleteNearestVehicle(range)
-- Params: range (optional, default: Config.DeleteVehicleRange)
-- Returns: boolean (true if vehicle was deleted)
exports('DeleteNearestVehicle', function(range)
    range = range or Config.DeleteVehicleRange

    local playerPed = PlayerPedId()
    local coords = GetEntityCoords(playerPed)
    local vehicle = GetClosestVehicleInRadius(coords, range)

    if vehicle and vehicle ~= 0 and DoesEntityExist(vehicle) then
        SetEntityAsMissionEntity(vehicle, true, true)
        DeleteVehicle(vehicle)
        return true
    end

    return false
end)

-- Get player's current vehicle
-- Usage: local vehicle = exports['PS-PyrosparkCore']:GetPlayerVehicle()
-- Returns: vehicle entity or 0 if not in vehicle
exports('GetPlayerVehicle', function()
    local playerPed = PlayerPedId()
    if IsPedInAnyVehicle(playerPed, false) then
        return GetVehiclePedIsIn(playerPed, false)
    end
    return 0
end)

-- Repair player's current vehicle
-- Usage: local success = exports['PS-PyrosparkCore']:RepairVehicle()
-- Returns: boolean (true if vehicle was repaired)
exports('RepairVehicle', function()
    local playerPed = PlayerPedId()

    if IsPedInAnyVehicle(playerPed, false) then
        local vehicle = GetVehiclePedIsIn(playerPed, false)

        SetVehicleFixed(vehicle)
        SetVehicleDeformationFixed(vehicle)
        SetVehicleUndriveable(vehicle, false)
        SetVehicleEngineOn(vehicle, true, true)
        SetVehicleDirtLevel(vehicle, 0.0)
        WashDecalsFromVehicle(vehicle, 1.0)

        return true
    end

    return false
end)
